% Code for: Capital Reallocation and Aggregate Productivity
% R. Cooper & I. Schott
% Code version: September 2022
function f = Fsys(x1,x0,y1,y0,PARMS,steady)

%% Extract values
mu_n1       = x1(1);            mu_n0   = x0(1);
phi_n1      = x1(2);            phi_n0  = x0(2);
A1          = exp(x1(3));       A0      = exp(x0(3));
K1          = x1(4);            K0      = x0(4);

k1          = y1(1:PARMS.N);    k0      = y0(1:PARMS.N);
pi1         = y1(PARMS.N+1);    pi0     = y0(PARMS.N+1);
C1          = y1(PARMS.N+2);    C0      = y0(PARMS.N+2);


% Steady state (long-run) values
pistar      = steady.pistar;
cstar       = steady.cstar;
k_a         = steady.k_a;
A           = 1; 
K           = steady.kstar;

%% What does the pi & k choice imply for mu_a and phi_a
sd          = PARMS.sigma;
mu          = -sd/2;
eps         = exp(mu + linspace(-3*sd,3*sd,PARMS.N)');
trans       = PARMS.PP;
clear sd mu

% Moments for adjustable plants
y_A0            = trans'*(eps.*(k0.^PARMS.ALPHA));
mu_a0           = trans'*(k0.^PARMS.ALPHA);
phi_a0          = y_A0 - PARMS.meaneps*mu_a0;
y_A1            = trans'*(eps.*(k1.^PARMS.ALPHA));
y_N0            = PARMS.meaneps * mu_n0 + phi_n0;
y_N1            = PARMS.meaneps * mu_n1 + phi_n1;

% Steady state values
y_A         = trans'*(eps.*(k_a.^PARMS.ALPHA));
mu_a        = trans'*(k_a.^PARMS.ALPHA);
phi_a       = y_A - PARMS.meaneps*mu_a;

%% FOC for pi0:

% Derivative of utility of current consumption w.r.t pi
dc0         = 1/C0 .* (A0*K0^PARMS.ALPHA*( PARMS.meaneps*(mu_a0 - mu_n0) + phi_a0 - phi_n0) - K0*pi0*(PARMS.F_upper));
% Marginal effect coming from the next period
dc1         = A1*K1^PARMS.ALPHA/C1  *(1-pi1)           *(PARMS.meaneps*(mu_a0 - mu_n0) +   PARMS.rho_eps   * (phi_a0 - phi_n0));
% Constant term once we are back in the steady state (mu_a = mu_n)
pi_ss_foc   = A*K^PARMS.ALPHA/cstar*(1-pistar)*(1-pi1)*                                   PARMS.rho_eps^2 * phi_a*((1-PARMS.rho_eps)/(1 - PARMS.rho_eps*(1-pistar))) /(1 - PARMS.B*PARMS.rho_eps*(1-pistar));

% FOC for pi0:
foc_pi0     = dc0 + PARMS.B*dc1 + PARMS.B^2*pi_ss_foc;

%% FOCs for capital

% Derivative of utility of current consumption with respect to capital (net of the k^(alpha - 1) term
dc_dk0      = A0*K0^PARMS.ALPHA/C0 * pi0*PARMS.ALPHA                   *(PARMS.eps*PARMS.rho_eps^0 + (1-PARMS.rho_eps^0)*PARMS.meaneps);
% Marginal effect coming from the next period
dc_dk1      = A1*K1^PARMS.ALPHA/C1 * pi0*PARMS.ALPHA*(1-pi1)           *(PARMS.eps*PARMS.rho_eps^1 + (1-PARMS.rho_eps^1)*PARMS.meaneps);
% Derivative of stream of future consumption (once we are back to the steady state)
k_ss_foc    = A*K^PARMS.ALPHA/C0  * pi0*PARMS.ALPHA*(1-pistar)*(1-pi1)*(PARMS.rho_eps^2*(PARMS.eps - PARMS.meaneps)/(1-PARMS.B*PARMS.rho_eps*(1-pistar)) + PARMS.meaneps/(1-PARMS.B*(1-pistar)) );
zeta        = dc_dk0 + PARMS.B*dc_dk1 + PARMS.B^2*k_ss_foc;
lambda      = (PARMS.PP'* (zeta.^(1/(1-PARMS.ALPHA))) )^(1-PARMS.ALPHA);

% FOC for capital
k_focs      = zeta.*k0.^(PARMS.ALPHA-1) - lambda;

%% FOC for aggregate capital

% Solow residual next period
SR1         = A1*(pi1*y_A1 + (1-pi1)*y_N1);
AC1         = pi1.^2/2*(PARMS.F_upper); % net of K
K_agg_FOC   = 1/C0 - PARMS.B * 1/C1 * (SR1*PARMS.ALPHA*K1^(PARMS.ALPHA-1) - AC1 + (1-PARMS.DELTA));

%% Resource constraint

SR0         = A0*(pi0*y_A0 + (1-pi0)*y_N0);
AC0         = pi0.^2/2*(PARMS.F_upper); % net of K
res_cnstr   = C0 + K1 - (SR0*K0^PARMS.ALPHA - AC0*K0 + (1-PARMS.DELTA)*K0);

%% Evolution of endogenous states

% Future moments for non-adjusters
muprime         = pi0*mu_a0 + (1-pi0)*mu_n0;
phiprime        = PARMS.rho_eps*(pi0*phi_a0 + (1-pi0)*phi_n0);

%% system of equations
f   =   [mu_n1 - muprime;
        phi_n1 - phiprime;
        k_focs;
        foc_pi0;     
        res_cnstr; % resource constraint
        K_agg_FOC; % FOC for agg capital
        log(A1) - PARMS.rhoA*log(A0)];



